<?php
/*
 * ----------------------------------------------------------------------
 *
 *                          Borlabs Cookie
 *                      developed by Borlabs
 *
 * ----------------------------------------------------------------------
 *
 * Copyright 2018 Borlabs - Benjamin A. Bornschein. All rights reserved.
 * This file may not be redistributed in whole or significant part.
 * Content of this file is protected by international copyright laws.
 *
 * ----------------- Borlabs Cookie IS NOT FREE SOFTWARE -----------------
 *
 * @copyright Borlabs - Benjamin A. Bornschein, https://borlabs.io
 * @author Benjamin A. Bornschein, Borlabs ben@borlabs.io
 *
 */

namespace BorlabsCookie\Cookie;

class Config
{
    private static $instance;

    private $config;

    public static function getInstance()
    {
        if (null === self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    public function __construct()
    {
        // Get all config values
        $this->loadConfig();
    }

    /**
     * loadConfig function.
     *
     * @access public
     * @param mixed $language (default: null)
     * @return void
     */
    public function loadConfig($language = null)
    {
        $this->config = $this->getConfig($language);

        return $this->config;
    }

    /**
     * getConfig function.
     *
     * @access public
     * @param mixed $language (default: null)
     * @return void
     */
    public function getConfig($language = null)
    {
        $config = [];

        if (empty($language)) {
            $configLanguage = Multilanguage::getInstance()->getCurrentLanguageCode();
        } else {
            $configLanguage = strtolower($language);
        }

        $config = get_option('BorlabsCookieConfig_'.$configLanguage, 'does not exist');

        if ($config === 'does not exist') {
            $config = $this->defaultConfig(true);
        }

        return $config;
    }

    /**
     * defaultConfig function.
     *
     * @access public
     * @param bool $installRoutine (default: false)
     * @return void
     */
    public function defaultConfig($installRoutine = false)
    {
        $defaultConfig = [
            'cookieActivated'=>'no',
            'testEnvironment'=>false,

            'codeCookieThirdParty'=>'',
            'googleAnalyticsId'=>'',
            'facebookPixelId'=>'',
            'cookieLifetime'=>365,
            'cookiesForBots'=>true,

            'showCookieOption2'=>true,
            'codeCookieFirstParty'=>'',
            'cookieLifetimeOption2'=>365,
            'executeFirstPartyCodeOnThirdParty'=>true,

            'showCookieOption3'=>true,
            'cookieLifetimeOption3'=>365,
            'cookieRespectDoNotTrack'=>false,

            'cookieBoxShow'=>'automatic',
            'cookieBoxBlocks'=>false,
            'cookieBoxIntegration'=>'script',
            'cookieBoxPosition'=>'top',

            'privacyPageId'=>0,
            'privacyPageTitle'=>'',
            'privacyPageURL'=>'',
            'imprintPageId'=>0,
            'imprintPageTitle'=>'',
            'imprintPageURL'=>'',
            'hideCookieBoxOnPage'=>[],

            'cookieDomain'=>'',
            'cookiePath'=>'/',
            'aggregateCookieConsent'=>false,

            'removeIframesInFeeds'=>true,
            'bctHostExceptions'=>[],

            'stylingPrimaryBackgroundColor'=>'#f9f9f9',
            'stylingSecondaryBackgroundColor'=>'#eeeeee',
            'stylingTextColor'=>'#606569',
            'stylingLinkColor'=>'#288fe9',
            'stylingSuccessMessageColor'=>'#5eb930',
            'stylingCookieOptionBorderColor'=>'#f2f2f2',

            'stylingCookieOptionBackgroundColor'=>'#ffffff',
            'stylingCookieOptionBackgroundColorHover'=>'#288fe9',
            'stylingCookieOptionTextColor'=>'#606569',
            'stylingCookieOptionTextColorHover'=>'#ffffff',

            'stylingCookieOptionBackgroundColor2'=>'#ffffff',
            'stylingCookieOptionBackgroundColorHover2'=>'#288fe9',
            'stylingCookieOptionTextColor2'=>'#606569',
            'stylingCookieOptionTextColorHover2'=>'#ffffff',

            'stylingCookieOptionBackgroundColor3'=>'#ffffff',
            'stylingCookieOptionBackgroundColorHover3'=>'#288fe9',
            'stylingCookieOptionTextColor3'=>'#606569',
            'stylingCookieOptionTextColorHover3'=>'#ffffff',

            'stylingBlockedContentBackgroundColor'=>'#efefef',
            'stylingBlockedContentTextColor'=>'#666666',
            'stylingBlockedContentButtonBackgroundColor'=>'#288fe9',
            'stylingBlockedContentButtonBackgroundColorHover'=>'#2875e9',
            'stylingBlockedContentButtonTextColor'=>'#ffffff',
            'stylingBlockedContentButtonTextColorHover'=>'#ffffff',
            'stylingBlockedContentSuccessMessage'=>'#5eb930',

            'stylingCustomCSS'=>'',
        ];

        // During the install routine the selected language is not available and it is always english
        if ($installRoutine === false) {
            $defaultConfig = array_merge(
                $defaultConfig,
                [
                'textCookieOptionThirdParty'=>_x('Accept all cookies', 'Cookie Message Box', 'borlabs-cookie'),
                'textCookieOptionFirstParty'=>_x('Accept first-party cookies only', 'Cookie Message Box', 'borlabs-cookie'),
                'textCookieOptionRejectThirdParty'=>_x('Reject all cookies', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpHeadline'=>_x('Help', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpTop'=>_x('To continue, you must make a cookie selection. Below is an explanation of the different options and their meaning.', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpCookieOptionThirdParty'=>_x('All cookies such as tracking and analytics cookies.', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpCookieOptionFirstParty'=>_x('Only cookies from this website.', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpCookieOptionRejectThirdParty'=>_x('No cookies except for those necessary for technical reasons are set. Borlabs Cookie already set a necessary cookie.', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpBottom'=>_x('You can change your cookie setting here anytime: <a href="%%privacy_url%%">%%privacy_title%%</a>. <a href="%%imprint_url%%">%%imprint_title%%</a>', 'Cookie Message Box', 'borlabs-cookie'),
                'textTitle'=>_x('Cookie Preference', 'Cookie Message Box', 'borlabs-cookie'),
                'textExplanation'=>_x('Please select an option. You can find more information about the consequences of your choice at <span data-borlabs-cookie-help-link>Help</span>. <a href="%%privacy_url%%">%%privacy_title%%</a> | <a href="%%imprint_url%%">%%imprint_title%%</a>', 'Cookie Message Box', 'borlabs-cookie'),
                'textInstruction'=>_x('Select an option to continue', 'Cookie Message Box', 'borlabs-cookie'),
                'textHelpLink'=>_x('More information', 'Cookie Message Box', 'borlabs-cookie'),
                'textBackLink'=>_x('Back', 'Cookie Message Box', 'borlabs-cookie'),
                'textCookieSettingSaved'=>_x('Your selection was saved!', 'Cookie Message Box', 'borlabs-cookie'),
                'textOptOut'=>_x('Click on the button below to opt-out of %%tracking_title%%.<br><a role="button">Opt-out</a>', 'Cookie Message Box', 'borlabs-cookie'),
                'textOptOutSuccess'=>_x('You\'ve successfully opted out from <strong>%%tracking_title%%</strong>.', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogColumnTitleUserAction'=>_x('User action', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogColumnTitleStamp'=>_x('Time stamp', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptionThirdParty'=>_x('All cookies accepted', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptionFirstParty'=>_x('Only first-party cookies accepted', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptionRejectAllCookies'=>_x('No cookies accepted', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptionNoSelection'=>_x('Website visited without user action', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptOutGoogleAnalytics'=>_x('Opted-out of Google Analytics', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptOutFacebookPixel'=>_x('Opted-out of Facebook Pixel', 'Cookie Message Box', 'borlabs-cookie'),
                'textLogOptInToUnblockType'=>_x('Always unblock %%name%%', 'Cookie Message Box', 'borlabs-cookie'),
                ]
            );
        }

        return $defaultConfig;
    }

    /**
     * get function.
     *
     * @access public
     * @param mixed $configKey (default: null)
     * @return void
     */
    public function get($configKey = null)
    {
        // Get complete config
        if (empty($configKey)) {
            if (!empty($this->config)) {
                return $this->config;
            } else {
                return false;
            }
        } else {
            if (isset($this->config[$configKey])) {
                return $this->config[$configKey];
            } else {
                // Fallback
                if (isset($this->defaultConfig()[$configKey])) {
                    return $this->defaultConfig()[$configKey];
                } else {
                    return false;
                }
            }
        }
    }

    /**
     * saveConfig function.
     *
     * @access public
     * @param mixed $configData
     * @return void
     */
    public function saveConfig($configData)
    {
        $configLanguage = Multilanguage::getInstance()->getCurrentLanguageCode();

        update_option('BorlabsCookieConfig_'.$configLanguage, $configData, 'no');

        $this->loadConfig();
    }
}
